﻿;***************************************************************************
;*
;* TESTSORT.ASM
;*
;* Sorting algorithm tests
;*
;* This program tests nine size-optimized sorting algorithms for 16-bit 8086
;* real mode.
;*
;* The sort functions are designed to run in the 'tiny' or 'small' memory
;* models and (with the exception of the 18-byte sort) callable from a
;* C language program[1].
;*
;* TestSort revisits the kind of assembly language size optimization that was
;* once important but has fallen into disuse outside of demo coding. My goal
;* was to keep the entirety of the final executable, both code and data, under
;* a kilobyte in size. Here are nine sort algorithms, with a test harness to
;* generate sample data, validate the correctness of each sort function, and
;* reports the results, in 998 bytes. I reached my goal so I stopped here but
;* if demo coding is your hobby there is more that can be squeezed and you
;* may enjoy seeing how much further you can take it.
;*
;* Refer to the accompanying README.TXT for additional details.
;*
;* [1] In the 'tiny' memory model all code and data fit within the same 64k
;*     memory segment and all segment registers address the same segment.
;*     The 'small' model stores code in one 64K segment and data in another
;*     64K segment. The sort functions work as written for either model.
;*     C pushes function arguments from right-to-left and the caller clears
;*     the stack. In 16-bit code, registers SI, DI, and BP are preserved.
;*     Functions return 16-bit values in AX, 32-bit values in DX:AX.
;*
;* Author:
;*
;*   David Stafford
;*
;* History:
;*
;*   1991 - 25-byte sort function (_sort25)
;*   2025 - Everything else
;*
;* Build tools:
;*
;*   Turbo Assembler version 4.1
;*     tasm.exe /l /ml /m9 /z testsort.asm
;*
;*   Turbo Linker version 7.1.30.1
;*     tlink /t testsort.obj
;*
;*   The executable COM file should build to exactly 998 bytes.
;*
;* MIT License:
;*
;*   https://opensource.org/license/mit
;*
;***************************************************************************


		title	TESTSORT

		locals	@@

		assume	cs:code, ds:code, es:code, ss:code

code		segment

		org	100h			;tiny model, .com file executable

;---------------------------------------------------------------------------
; Code execution entry point
;---------------------------------------------------------------------------

start		proc	near

IFDEF UNUSED
		cld				;we can assume it is clear on startup
ENDIF
		call	writeln_ip
		db	'Sort algorithm tests'
s_crlf:		db	13,10,0			;used by write_newline

		; Make sure we have enough memory for the stack

		cmp	sp,offset stack_top ;do we have enough memory?
		jbe	@@oom

		; Test the fault-injection functions

		call	writeln_ip
		db	'Fault injection tests should FAIL',0

		mov	ax,offset fault_table
		push	ax
		call	test_sorts

		; Test the sort algorithms

		call	writeln_ip
		db	13,10,'Sort algorithm tests should PASS',0

		mov	ax,offset sort_table
		push	ax
		call	test_sorts

		; All done, exit

		call	writeln_ip
		db	13,10,'Tests complete',0

		ret

@@oom:		call	writeln_ip
		db	'Error: Insufficient memory',0

		ret

		endp


;---------------------------------------------------------------------------
; pascal void fill_array( unsigned multiplier, unsigned increment, int count )
;
; Fills the static array with a desired pattern to test a sort algorithm.
;
; Flex a simple pseudo-random number generator to populate the array with
; integers that are constant, increasing, decreasing, or pseudo-random,
; by careful choices for the multiplier and increment. This function also
; bookends the array with sentinel values which can be checked to detect
; errors due to underflow or overflow and finally stores a simple checksum
; to catch corruption of the array contents.
;
; In:
;
;  multiplier: multiplier for the random number generator
;  increment:  increment for the random number generator
;  count:      number of integers in the array
;
; Out:
;
;  The static array is filled as requested
;  Array header and footer values are in place
;  Array checksum is in place
;---------------------------------------------------------------------------

fill_array	proc	near

		push	di

		mov	di,offset array_header
		mov	ax,7fffh		;header sentinel value
		stosw

		xor	ax,ax			;state = 0
		push	ax			;checksum on the stack
		mov	bx,sp

		; This loop intentionally runs one past the array length, safely,
		; putting a value into the footer which we will fix up at the end.
		; This avoids testing for the zero-length condition before entering
		; the fill loop and results in a known value in the footer (register
		; ax) which we can exploit to zero it out later.
		;
		; Since the checksum and state both start at zero, the first checksum
		; update has no effect, and the loop exits after the final overwrite
		; into the footer, without updating the checksum with this value.
		;
		; The core of the loop is a simple linear congruential generator
		; which can be used to generate different patterns of data depending
		; on the two values for the multiplier and increment:
		; https://en.wikipedia.org/wiki/Linear_congruential_generator

@@next:		add	[bx],ax			;checksum = checksum + state
		mul	word ptr [bx+10]	;state = state * multiplier
		add	ax,[bx+8]		;state = state + increment
		stosw

@@bottom:	dec	word ptr [bx+6]		;length
		jge	@@next

@@finish:	xor	[di-2],ax		;zero out the footer
		pop	[di]			;pop the checksum and
						;save it after the footer
		pop	di
		ret	6

		endp


;---------------------------------------------------------------------------
; pascal int check_array( int count)
;
; Makes these checks:
;
; 1. Verifies the presence of the array header (7fffh) and footer (0) which
;    are stored just ahead of and immediately following the array data.
; 2. Verifies the data checksum that follows the footer. The checksum is a
;    simple summation of the array data.
; 3. Verifies the array is sorted in ascending order.
;
; In:
;
;  count: number of integers in the array
;
; Out:
;
;  zero if the array is sorted and the header/footer/checksum are valid
;  non-zero on any error
;---------------------------------------------------------------------------

check_array	proc	near

		pop	ax			;return address
                pop	cx			;get count
                push	ax			;restore return
		push	si

		mov	si,offset array_header
		lodsw				;get header value
		cmp	ax,7fffh		;(1) check for max signed int
		jne	@@fail			;not a match? fail

		; The next two instructions depend on ax==7fffh

		cwd				;(1)(2) init the checksum in dx
		inc	ax			;(1) ax=-32768 to prime prev int

		; The following loop is designed to run one integer past the
		; end of the array so that ax, on completion, will contain
		; the value of the footer (should be 0) and si will point to
		; the checksum.

@@next:		xchg	ax,bx			;bx=prev int
		lodsw				;ax=this int
		dec	cx			;count--
		jl	@@finish		;all done?
		add	dx,ax			;(2) update the checksum
		cmp	ax,bx			;this int >= prev int?
		jge	@@next			;yes, continue

		; If we get here, an array element was out-of-order,
		; so we will make sure that dx (the checksum) will be
		; non-zero going into the final 'or' which will signal
		; an error condition.

@@fail:		db	0bah			;(2) mov dx,...

		; If we get here, the array loop above completed
		; normally and si points to the checksum. Subtract the
		; stored checksum from the computed checksum in dx.
		; If they match, dx will equal zero. Note that this
		; instruction may be masked by the previous instruction
		; at @@fail which loads dx with a non-zero value to
		; signal an error on function return.

@@finish:	dw	142bh			;(2) sub dx,[si]

		; At this point, if all is well, ax contains the footer,
		; which should be zero, and dx contains the computed checksum
		; minus the actual checksum, which should also be zero. If
		; either one of these registers is not zero, there has been an
		; error. Simply 'or' them together to create the return code.

		or	ax,dx			;(2)

		pop	si
		ret

                endp


;---------------------------------------------------------------------------
; pascal void test_sorts( void *sort_table )
;
; Iterates through each sort function to test.
;
; In:
;
;   sort_table: pointer to a table of structures where each item contains:
;               1. a pointer to a string (the name)
;               2. a pointer to a function
;               See sort_table and fault_table for examples.
;
; Out:
;
;   1. displays the function name
;   2. runs the function through a battery of tests
;   3. shows the pass/fail count results
;---------------------------------------------------------------------------

test_sorts	proc	near
		pop	bx			;return address
		pop	ax			;get sort_table
		push	bx			;restore return

		push	si
		xchg	ax,si

@@next:		lodsw				;sort function name
		xchg	ax,cx
		jcxz	@@exit			;null, end-of-list?

		push	cx			;(1) sort function name

		call	write_ip
		db	13,10,'Test: ',0

		call	write			;(1) sort function name

		; There are a couple of size optimizations here that deserve an
		; explanation.
		;
		; 1. write() returns zero in dx by design. We take advantage
		;    of that to create a couple of locals on the stack for
		;    pass/fail counts. These are later consumed as parameters
		;    in two calls to write_uint() below.
		;
		; 2. The pass/fail locals are updated in test_fills(), two
		;    levels down the stack. test_lengths() and test_fills()
		;    are treated as a couple of nested functions for the
		;    purpose of accessing these locals. Just be aware of this
		;    and be careful when calculating the stack offsets to
		;    access them, if anything should change.

                push	dx			;(2) fail count
                push	dx			;(3) pass count

		lodsw
		push	ax			;sort function ptr
		call	test_lengths

		call	write_ip
		db	'Pass=',0

		call	write_uint		;(3) pass count

		call	write_ip
		db	',Fail=',0

		call	write_uint		;(2) fail count

		jmp	@@next

@@exit:		pop	si
		jmp	write_newline		;return from write_newline()

		endp


;---------------------------------------------------------------------------
; pascal void test_lengths( (*sortfn)() )
;
; Given a sort function, this iterates through each array length to test.
; It is table-driven by count_table in the data section.
;
; In:
;
;   sortfn: pointer to a C-callable sort function
;
; Out:
;
;   nothing
;---------------------------------------------------------------------------

test_lengths	proc	near

		pop	ax			;return address
		pop	dx			;sort function ptr
		push	ax			;restore return

		push	si
		mov	si,offset count_table

@@next:		push	dx			;(1) save dx
		push	dx			;sort function ptr
		lodsw				;get length
		push	ax			;length
		call	test_fills
		pop	dx			;(1) restore dx

		cmp	si,offset count_table_end
		jb	@@next

		pop	si
		ret

		endp


;---------------------------------------------------------------------------
; pascal void test_fills( (*sortfn)(), int count )
;
; Given a sort function and array length, iterate through every array fill
; method to test. This is table-driven by fill_table in the data section.
; This function fills the array with a data pattern, calls the sort function,
; checks the array for correctness, and updates the pass/fail counts.
;
; In:
;
;   sortfn: pointer to a C-callable sort function
;   count:  number of integers in the array
;
; Out:
;
;   pass/fail counts are updated
;---------------------------------------------------------------------------

test_fills	proc	near

		pop	dx			;return address
		pop	ax			;count of integers
		pop	cx			;sort function
		push	dx			;restore return

		push	di
		xchg	ax,di			;di=count of integers

		push	bp
		mov	bp,cx			;bp=sort function

		push	si
		mov	si,offset fill_table

		; It's not good style to mix I/O with computation like
		; this but we are optimizing for size and it's smaller.

@@next:		call	write_ip
		db	'.',0

		lodsw				;multiplier
		push	ax
		lodsw				;increment
		push	ax
		push	di			;count of integers
		call	fill_array

		mov	ax,offset array
		push	ax			;the array
		push	di			;count of integers
		call	bp			;call the sort function
		pop	ax			;C-callable so clear the stack
		pop	ax

		push	di			;count of integers
		call	check_array
		xchg	ax,cx			;result is in cx

		; When we get here, cx contains the pass/fail result from
		; validating the behavior of the sort function:
		; cx == 0 for success
		; cx != 0 for failure
		;
		; Update the pass/fail counts that are higher up on the stack
		; from test_sorts(). If anything should change that affects the
		; stack position, you must adjust the offset for bx, below.

		mov	bx,sp
                add	bx,14			;bx points to pass count
                jcxz	@@done			;success?
                inc	bx			;move up the stack to fail count
                inc	bx

@@done:		inc	word ptr [bx]		;bump the appropriate counter

		cmp	si,offset fill_table_end
		jb	@@next

		pop	si
		pop	bp
		pop	di
		ret

		endp


;---------------------------------------------------------------------------
; These are a collection of output functions for strings, newlines, and
; unsigned integers.
;
; void pascal write( char *str )
; Outputs a string on stdout
;
; void pascal writeln( char *str )
; Outputs a string and appends a newline (cr/lf)
;
; void pascal write_ip()
; Outputs a string embedded in code that immediately follows the call to write_ip()
;
; void pascal writeln_ip()
; Appends a newline after a write_ip()
;
; void pascal write_bx()
; Outputs the string addressed by the "bx" register
;
; void pascal write_newline()
; Outputs a newline (cr/lf)
;
; void pascal write_uint( unsigned value )
; Outputs an unsigned integer
;
; In:
;
;  See the individual functions for parameters
;
; Out:
;
;  The selected data is sent to the output device (usually the console)
;  dx=0 for all functions
;---------------------------------------------------------------------------

write_procs	proc	near

		; writeln() is not presently in use and can be enabled if desired.

IFDEF UNUSED
writeln:	pop	cx			;(1) cx=return address
		call	write			;write original string
		push	cx			;(1) restore return address
		jmp	write_newline
ENDIF

writeln_ip:	call	write
		push	bx			;get the return address

write_newline:	mov	bx,offset s_crlf
		jmp	write_bx

write_ip:	call	write
		jmp	bx			;simulate a return

write:		pop	ax			;return address
		pop	bx			;bx=string to output
		push	ax			;restore return

		; This is the central output loop, shared by all functions.
		;
		; The code is structured with these goals in mind:
		;
		; 1. The first iteration skips the "int 21", eliminating an exit
		;    test from inside the loop.
		; 2. bx ends one byte past the final character which is needed by
		;    write_ip() and writeln_ip() who use bx as the return address.
		; 3. cx is preserved. This is helpful for writeln().
		; 4. dx is zeroed on exit (a minor benefit for some callers).
		;
		; None of these are really vital but they save code elsewhere and
		; it was possible to attain them all at no additional cost to the
		; size of this loop.

		db	0b8h			;"mov ax,21cdh" to mask the "int 21h"
@@print_loop:	int	21h
write_bx:	mov	ah,02h			;DOS function to output a single char
		cwd				;dx=0 so we can "xor dl" below
		inc	bx			;bump pointer
		xor	dl,[bx-1]		;fetch a character, set the zero flag
		jnz	@@print_loop		;not 0, end-of-string?
                ret				;all write_procs functions pass through here

		; Converts an unsigned integer to a string by walking backwards
		; from the end of the buffer, peeling off the least-significant
		; digits until none are left. This loop executes at least once,
		; so '0' is handled correctly. On exit, bx points to the final
		; stored digit (at the lowest memory address) which is convenient
		; for write_bx() to display the string.

write_uint:	pop	dx			;return address
		pop	cx			;cx=the integer
		push	dx			;restore return

		mov	bx,offset strbuf_end-1	;bx=end of the string buffer
		mov	ax,10			;base 10
		mov	[bx],ah			;trailing zero

@@digit_loop:	cwd				;dx=0, prep for divide
		xchg	ax,cx			;cd=10
		div	cx			;ax=ax/10, dl=ax%10
		add	dl,'0'			;convert to ascii
		dec	bx			;dec the pointer
		mov	[bx],dl			;store the digit
		xchg	ax,cx
		jcxz	write_bx
		jmp	@@digit_loop

		endp


;---------------------------------------------------------------------------
; These are a collection of C-callable functions which create arrays with
; different types of faults. They follow the same convention as the sort
; functions and can be used interchangeably. Their purpose is to validate
; the check_array() function by deliberately injecting errors in the array
; metadata as well as the contents of the array itself.
;
; cdecl void bad_order( int count, int *array )
; Ensures the array elements are not in ascending order.
;
; cdecl void bad_header( int count, int *array )
; Ensures the magic value in the header fails to match.
;
; cdecl void bad_checksum( int count, int *array )
; Ensures the stored checksum will not match the array data.
;
; cdecl void bad_footer( int count, int *array )
; Ensures the magic value in the footer fails to match.
;
; In:   array to sort
;       count of integers
;
; Out:  the array will fail to be validated by check_array()
;---------------------------------------------------------------------------

fault_procs	proc	near

@@helper:	mov	bx,sp
		push	[bx+6]			;(1) array pointer
		push	[bx+4]			;(2) count

		; We will start with a known-good array and then break it.
		; _sort23 leaves the parameters on the stack unchanged which
		; we can take advantage of to recover the array pointer and
		; count, reducing code size.

		call	_sort23			;call a known-good sort

		pop	ax			;(2) ax=count
		pop	bx			;(1) bx=array pointer
		ret

_bad_order:	call	@@helper

		; There is an edge case when the array contains fewer than two
		; integers: The order can't be broken. A more complex program
		; might deal with this in a different way but for our purposes
		; it is enough to fall back into breaking the header so the
		; validate in check_array() will fail in every case.

		dec	ax			;count
		jle	@@break_header		;was count < 2?

		; The simplest way to ensure the array fails the order test
		; is to sit in a loop, increment the first item and decrementing
		; the second, until the first is greater than the second.

@@top:		inc	word ptr [bx]		;increment first integer
		dec	word ptr [bx+2]		;decrement second integer
		mov	ax,[bx]
		cmp	ax,[bx+2]
		jle	@@top			;still in-order?
		ret

_bad_header:	call	@@helper
@@break_header:	dec	bx			;bx points into the header now
		jmp	@@break2

_bad_checksum:	call	@@helper
		inc	ax			;inflate the count to cause
		jmp	@@break1		;bx to address the checksum

_bad_footer:	call	@@helper
@@break1:	add	bx,ax			;move bx to the footer
		add	bx,ax
@@break2:	inc	byte ptr [bx]		;break whatever bx points to
		ret

		endp


;---------------------------------------------------------------------------
; cdecl void sort25( int count, int *array )
;
; Sorts an array of ints. C-callable. Time complexity is quadratic (n^2).
;
; This is the original sort I shared in 1991.
;
; Size: 25 bytes
;
; In:   array to sort
;       count of integers
;
; Out:  array is sorted
;---------------------------------------------------------------------------

            	public	_sort25

sort25_proc	proc	near

@@top:		mov	dx,[bx]			;swap two adjacent integers
		xchg	dx,[bx+2]
		xchg	dx,[bx]
		cmp	dx,[bx]			;did we put them in the right order?
		jl	@@top			;no, swap them back
		inc	bx			;bump the pointer
		inc	bx
		loop	@@top

_sort25:	pop	dx			;get return address (entry point)
		pop	cx			;get count
		pop	bx			;get array
		push	bx			;save array
		dec	cx			;(1) decrement count
            	push	cx			;save count
            	push	dx			;save return address

		jg	@@top			;(1) at least two to sort?

		ret

		endp


;---------------------------------------------------------------------------
; cdecl void sort25a( int count, int *array )
;
; Sorts an array of ints. C-callable. Time complexity is quadratic (n^2).
;
; This is my favorite of the 25-byte quadratic C-callable sorts. It is
; simpler and prettier code than my version from 1991.
;
; Size: 25 bytes
;
; In:   array to sort
;       count of integers
;
; Out:  array is sorted
;---------------------------------------------------------------------------

		public	_sort25a

sort25a_proc	proc	near

@@top:		mov	ax,[bx]			;get first integer of pair
		inc	bx			;bump the pointer
                inc	bx
                cmp	ax,[bx]			;compare to second integer
                jle	@@skip			;no need to swap them?

		xchg	ax,[bx]			;swap adjacent pair
		xchg	ax,[bx-2]

@@skip:		loop	@@top

_sort25a:	pop	ax			;get return address (entry point)
		pop	cx			;cx=count
		pop	bx			;bx=array
		push	bx			;save array
		dec	cx			;(1) decrement count
		push	cx			;save count
		push	ax			;save return address

		jg	@@top			;(1) at least two to sort?

                ret

                endp


;---------------------------------------------------------------------------
; cdecl void sort25b( int count, int *array )
;
; Sorts an array of ints. C-callable. Time complexity is cubic (n^3).
;
; I brushed right up against the insight that eventually led to the 23-byte
; sort but failed to make the connection at the time. It shows the fall-through
; idea to restart after the swap but the main loop is at the bottom with two
; jumps weighing heavily. Compare this to sort23 and see how close these are.
; I had to thrash about a bit with some false starts before coming back to
; this pattern, eliminating one of the jumps by simply relocating the loop.
;
; There is no reason to ever use this sort. It is 25 bytes and runs in cubic
; time. But it was a necessary step on my way to finding something better.
;
; Size: 25 bytes
;
; In:   array to sort
;       count of integers
;
; Out:  array is sorted
;---------------------------------------------------------------------------

		public	_sort25b

sort25b_proc	proc	near

@@swap:		xchg	ax,[bx]			;swap adjacent pair of integers
		xchg	ax,[bx-2]

_sort25b:	pop	ax			;get return address (entry point)
		pop	cx			;cx=count
		pop	bx			;bx=array
		push	bx			;save array
		push	cx			;save count
		push	ax			;save return address

		dec	cx			;decrement count
		jle	@@exit			;less than two?

		; Loop through the array, looking for a pair of integers to swap.
		; If we find them, do the swap, and start over from the beginning.

@@comp:		mov	ax,[bx]			;get first integer of pair
		inc	bx			;bump the pointer
		inc	bx
		cmp	ax,[bx]			;compare to second integer
		jg	@@swap			;out-of-order? swap them
		loop	@@comp

@@exit:		ret

		endp


;---------------------------------------------------------------------------
; cdecl void sort25c( int count, int *array )
;
; Sorts an array of ints. C-callable. Time complexity is quadratic (n^2).
;
; Size: 25 bytes
;
; In:   array to sort
;       count of integers
;
; Out:  array is sorted
;---------------------------------------------------------------------------

		public	_sort25c

sort25c_proc	proc	near

@@top:		xchg	ax,[bx+2]		;(1) ax=the second integer, [bx+2]=garbage
@@back:		xchg	ax,[bx]			;swap the second with the first integer
		cmp	ax,[bx]			;do we have the larger one in ax now?
		jl	@@back			;no, swap them back
		inc	bx			;bump the pointer
		inc	bx
		mov	[bx],ax			;(1) patch the second integer, now at [bx]
		loop	@@top

_sort25c:	pop	ax			;get return address (entry point)
		pop	cx			;cx=count
		pop	bx			;bx=array
		push	bx			;save array
		dec	cx			;(2) decrement count
		push	cx			;save count
		push	ax			;save return address

		jg	@@top			;(2) at least two to sort?

		ret

		endp


;---------------------------------------------------------------------------
; cdecl void sort29( int count, int *array )
;
; Sorts an array of ints. C-callable. Time complexity is quadratic (n^2).
;
; Size: 29 bytes
;
; In:   array to sort
;       count of integers
;
; Out:  array is sorted
;---------------------------------------------------------------------------

		public	_sort29

sort29_proc	proc	near

@@next:		cmp	[bx],dx			;found the largest so far?
		jl	@@around		;no, skip it

		db	0b0h			;"mov al,..." to mask the next instruction
@@entry:	push	bx			;(1) save the array start address
		xchg	[bx],dx			;swap the current value with the carrying register

@@around:	inc	bx			;bump the pointer
		inc	bx

		dec	cx			;end of the array?
		jge	@@next			;no, continue

		xchg	[bx-2],dx		;patch last array element
		pop	bx			;(1) restore the array starting address
		xchg	[bx],dx			;patch first array element

_sort29:	pop	ax			;get return address (entry point)
		pop	cx			;cx=count
		pop	bx			;bx=array
		push	bx			;save array
		dec	cx			;(2) decrement count
		push	cx			;save count
		push	ax			;save return address

		jg	@@entry			;(2) at least two to sort?

		ret

		endp


;---------------------------------------------------------------------------
; cdecl void sort28a( int count, int *array )
;
; Sorts an array of ints. C-callable. Time complexity is quadratic (n^2).
;
; Size: 28 bytes
;
; In:   array to sort
;       count of integers
;
; Out:  array is sorted
;---------------------------------------------------------------------------

		public	_sort28a

sort28a_proc	proc	near

@@next:		inc	bx			;bump the pointer
		inc	bx
@@compare:	cmp	[bx],dx			;(1) found the largest so far?

		db	0b0h			;"mov al,..." to mask the next instruction
@@entry:	push	bx			;(2) save the array start address

		xchg	[bx],dx			;swap the current value with the carrying register
		jl	@@compare		;(1) this is only affected by the above compare
						;and not the "dec cx" at (3)

		dec	cx			;end of the array?
		jge	@@next			;no, continue

		; When we get here we have the largest value from the array in dx.
		; The first array element has a garbage value and must be patched.
		; We swap dx with the last element which will then be swapped into
		; the first element to patch up the array.

		xchg	[bx],dx			;patch last array element
		pop	bx			;(2) restore the array starting address
		xchg	[bx],dx			;patch first array element

_sort28a:	pop	ax			;get return address (entry point)
		pop	cx			;cx=count
		pop	bx			;bx=array
		push	bx			;save array
		dec	cx			;(3) decrement count
		push	cx			;save count
		push	ax			;save return address

		jg	@@entry			;(3) at least two to sort?

		ret

		endp


;---------------------------------------------------------------------------
; cdecl void sort28b( int count, int *array )
;
; Sorts an array of ints. C-callable. Time complexity is quadratic (n^2).
;
; Size: 28 bytes
;
; In:   array to sort
;       count of integers
;
; Out:  array is sorted
;---------------------------------------------------------------------------

		public	_sort28b

sort28b_proc	proc	near

@@next:		inc	bx			;bump pointer
		inc	bx
		cmp	[bx],dx			;found the largest so far?
		jle	@@around		;no, skip it

		; When we get here we have found the largest value seen so far.
		; Copy it into dx and copy the address into ax so, when we get
		; to the end of the array, we can swap the value at the saved
		; location with the value at the final array element.
		;
		; This is also the entry point into the inner loop so the
		; largest value and the saved location are initialized here.

@@entry:	mov	ax,bx			;(1) save this address
		mov	dx,[bx]			;update the largest value

@@around:	dec	cx			;decrement counter
		jge	@@next			;loop for the next integer

		; When we get here we have exhausted the array and bx points
		; to the last element. Swap the last element with the one that
		; was saved (which contained the largest value.)

		xchg	dx,[bx]			;patch last array item
		xchg	ax,bx			;(1) get the saved address
		xchg	dx,[bx]			;patch the previous largest item

_sort28b:	pop	ax			;get return address (entry point)
		pop	cx			;cx=count
		pop	bx			;bx=array
		push	bx			;save array
		dec	cx			;(2) decrement count
		push	cx			;save count
		push	ax			;save return address

		jg	@@entry			;(2) at least two to sort?

		ret

		endp


;---------------------------------------------------------------------------
; cdecl void sort23( int count, int *array )
;
; Sorts an array of ints. C-callable.
;
; Caution: This sort has cubic (n^3) performance. It's fine for
;          a few hundred integers. Beware of sorting thousands.
;
; Size: 23 bytes
;
; In:   array to sort
;       count of integers
;
; Out:  array is sorted
;---------------------------------------------------------------------------

		public	_sort23

sort23_proc	proc	near

@@top:        	mov	ax,[bx]			;get the first integer of a pair
            	inc	bx			;bump pointer
            	inc	bx
            	cmp	ax,[bx]			;compare to the second integer
            	jle	@@bottom		;no swap? continue the loop

		xchg	ax,[bx]			;swap the integer pair
                xchg	ax,[bx-2]

		; If we get here, there has been a swap, and we do not continue
		; looping over the array. Instead, the code path falls through to
		; the entry point which starts everything over from the beginning.

_sort23:	pop	ax			;get return address (entry point)
            	pop	cx			;cx=count
            	pop	bx			;bx=array
            	push	bx			;save array
      		push	cx			;save count
		push	ax			;save return address

@@bottom:	dec	cx			;decrement count
		jg	@@top			;at least two to sort?

		ret

		endp


;---------------------------------------------------------------------------
; cdecl void wrap_sort18( int count, int *array )
;
; A C-callable wrapper around _sort18 for the convenience of our testing
; functions (which assume C-callability.)
;
; In:  array to sort
;      count of integers
;
; Out: array is sorted
;---------------------------------------------------------------------------

_wrap_sort18	proc	near

		pop	ax			;ax=return address
		pop	dx			;dx=count
		pop	bx			;bx=array
		push	bx			;save array
		push	dx			;save count
		push	ax			;save return address

		push	si			;save si
		call	_sort18
		pop	si			;restore si

		ret

		endp


;---------------------------------------------------------------------------
; Sorts an array of ints. NOT C-callable.
;
; Same algorithm as _sort23 but optimized for size by removing the constraint
; of following a calling convention. This reduces the code to its purest and
; simplest form. As with _sort23, the runtime complexity is cubic, and sorting
; more than a few hundred items is not recommended.
;
; Size:      18 bytes
;
; In:        bx = array to sort
;            dx = count of integers
;            direction-flag = clear
;
; Out:       array at bx is sorted
;
; Preserves: bx, dx, bp, di
;
; Destroys:  ax, cx, si
;---------------------------------------------------------------------------

		public	_sort18

sort18_proc	proc	near

@@top:		lodsw				;fetch integer
		cmp	ax,[si]			;compare to next integer
		jle	@@bottom		;no swap? continue the loop

		xchg	ax,[si]			;swap the integer pair
		xchg	ax,[si-2]

		; If we get here, there has been a swap, and we do not continue
		; looping over the array. Instead, the code path falls through to
		; the entry point which starts everything over from the beginning.

_sort18:	mov	si,bx			;si=array (entry point)
		mov	cx,dx			;cx=count

@@bottom:	dec	cx			;decrement count
		jg	@@top			;at least two to sort?

		ret

		endp


;---------------------------------------------------------------------------
; Initialized data section
;---------------------------------------------------------------------------

; Fault injection table. Each entry is a tuple:
; 1. Pointer to the name of the algorithm (for displaying).
; 2. Pointer to the fault injection function.
; Used by test_sorts()

fault_table	dw	offset s_bad_order,    offset _bad_order
		dw	offset s_bad_header,   offset _bad_header
		dw	offset s_bad_footer,   offset _bad_footer
		dw	offset s_bad_checksum, offset _bad_checksum
		dw	0

; Sorting algorithm table. Each entry is a tuple:
; 1. Pointer to the name of the algorithm (for displaying).
; 2. Pointer to the sort function.
; Used by test_sorts()

sort_table	dw	offset s_sort25_1991,  offset _sort25
                dw	offset s_sort25a_2025, offset _sort25a
		dw	offset s_sort25b_2025, offset _sort25b
                dw	offset s_sort25c_2025, offset _sort25c
                dw	offset s_sort29_2025,  offset _sort29
                dw	offset s_sort28a_2025, offset _sort28a
                dw	offset s_sort28b_2025, offset _sort28b
		dw	offset s_sort23_2025,  offset _sort23
		dw	offset s_sort18_2025,  offset _wrap_sort18
		dw	0

; Array fill method table. Each entry is a tuple:
; 1. Pointer to the name of the method (for displaying).
; 2. The multiplier for the generator.
; 3. The increment for the generator.
; The multipliers for the random fills are chosen to have 64k periods.
; Used by test_fills()

fill_table	dw	    0, -32768		;s = (0 * s) - 32768;     Fill with -32768
		dw	    0,      0		;s = (0 * s) + 0;         Fill with zeros
		dw	    0, +32767		;s = (0 * s) + 32767;     Fill with +32767
		dw	    1,      1		;s = (1 * s) + 1;         Fill with increasing values
		dw	    1,     -1		;s = (1 * s) - 1;         Fill with decreasing values
		dw	12829,  47989		;s = (12829 * s) + 47989; Fill with random numbers
fill_table_end	equ	$

; Array length table. Each entry is an array length for testing the sort functions.
; The array is physically located in uninitialized data. BE CAREFUL to ensure the
; largest length listed here fits within the space budgeted there.
; Used by test_lengths()

count_table	dw	0,1,2,3,100,500,1000
count_table_end	equ	$

; Strings

s_bad_order	db	'Bad order',0
s_bad_header	db	'Bad header',0
s_bad_footer	db	'Bad footer',0
s_bad_checksum	db	'Bad checksum',0
s_sort25_1991	db	'1991 25 sort',0
s_sort25a_2025	db	'2025 25a sort',0
s_sort25b_2025	db	'2025 25b sort',0
s_sort25c_2025	db	'2025 25c sort',0
s_sort29_2025	db	'2025 29 sort',0
s_sort28a_2025	db	'2025 28a sort',0
s_sort28b_2025	db	'2025 28b sort',0
s_sort23_2025	db	'2025 23 sort',0
s_sort18_2025	db	'2025 18 sort',0

;---------------------------------------------------------------------------
; Unitialized data
;---------------------------------------------------------------------------

strbuf		equ	$
strbuf_end	equ	strbuf+128		;128 bytes for a string buffer

array_header	equ	strbuf_end
array		equ	array_header+2		;room for SENTINEL_HEADER
array_footer	equ	array+(10000*2)		;room for 10,000 integers
array_checksum	equ	array_footer+2		;room for SENTINEL_FOOTER
array_end	equ	array_checksum+2	;room for array_checksum

;---------------------------------------------------------------------------
; Stack
;
; The stack is not physically relocated here. This is a space reserved after
; the end of unitialized data and checked against the stack pointer on startup
; to ensure enough room is available for the stack to grow.
;---------------------------------------------------------------------------

stack_bottom	equ	array_end
stack_top	equ	stack_bottom+1024	;one kilobyte reserved

code        	ends

;---------------------------------------------------------------------------
; End of program
;---------------------------------------------------------------------------

		end	start

